/* ***************************************************************************+
 * ITX package (cnrg.itx) for telephony application programming.              *
 * Copyright (c) 1999  Cornell University, Ithaca NY                          *
 *                                                                            *
 * This program is free software; you can redistribute it and/or modify       *
 * it under the terms of the GNU General Public Liense as published by        *
 * the Free Software Foundation, either version 2 of the License, or (at      * 
 * your option) any later version.                                            *
 *                                                                            *
 * The ITX package is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY, without even the implied warranty of MERCHANTABILITY *
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License   *
 * for more details.                                                          * 
 *                                                                            *
 * A copy of the license is distributed with this package.  Look in the docs  *
 * directory, filename GPL.                                                   *
 *                                                                            * 
 * Contact information:                                                       *
 * Donna Bergmark                                                             *
 * 484 Rhodes Hall                                                            *
 * Cornell University                                                         *
 * Ithaca, NY 14853-3801                                                      *
 * bergmark@cs.cornell.edu                                                    *
 ******************************************************************************/
package shared;

import cnrg.itx.datax.*;

/**
 * A class representing a destination observer.  A <code>DestinationObserver</code> simply
 * records notable statistics concerning the <code>Channel</code> that it is attached to.
 * 
 * @author Jason Howes
 * @version 1.0, 2/10/1999
 * @see cnrg.itx.datax.Channel
 * @see cnrg.itx.datax.Destination
 * @see cnrg.itx.datax.Statistics
 * @see cnrg.itx.datax.Stats
 */

public class DestinationObserver implements Destination
{	
	/**
	 * Holds the current statistics of the <code>Channel</code>.
	 */
	private Stats mCurrentStats;
	
	/**
	 * Is the destination open?
	 */
	private boolean mOpen;
	
	/**
	 * Is the destination muted?
	 */
	private boolean mMuted;
	
	/**
	 * Current stream offset
	 */
	private long mStreamOffset;
	
	/**
	 * Current channel byte offset key.
	 */
	public static final String BYTE_OFFSET = new String("DO_BYTE_OFFSET");
	
	/**
	 * Class constructor.
	 * 
	 * @exception <code>DataException</code> on error.
	 */
	public DestinationObserver() throws DataException
	{
		mCurrentStats = new Stats();
		mStreamOffset = 0;
		mCurrentStats.addStat(BYTE_OFFSET, new Long(mStreamOffset));
		mOpen = true;
		mMuted = false;
	}
	
   /**
    * Returns a collection of properties supported.
    * 
    * @exception DataException thrown when the properties in the device are incompatible
    */ 
   public PropertiesCollection getProperties() throws DataException
   {
	   return new PropertiesCollection();
   }
   
   /**
    * Function to set the given properties collection into the device. Works under the assumption
    * that the properties collection was generated LOCALLY and is a request for options.
    * 
    * @exception DataException thrown when trying to set incompatible properties collections
    */
   public void setProperties(PropertiesCollection pc) throws DataException
   {
   }

   /**
    * Function to set the given properties collection into the device. Works under the 
    * assumption that this is the properties collection of the peer.
    */
   public void setPeerProperties(PropertiesCollection pc) throws DataException
   {
   }
	
	/**
	 * Writes to the <code>DestinationObserver</code>.
	 * 
	 * @param b byte array of data to write.
	 * @exception <code>DataException</code> on write error.
	 */
	public void write(byte []b) throws DataException
	{
		if (mOpen)
		{
			incrementStreamOffset(b.length);
			Long offset = new Long(getStreamOffset());
			mCurrentStats.addStat(BYTE_OFFSET, offset);
		}
	}
	
	/**
	 * Closes the <code>DestinationObserver</code>.
	 */	
	public void close()
	{
		reset();
		mOpen = false;
	}
	
	/**
	 * Mutes the <code>DestinationObserver</code>.
	 * 
	 * @param mute mute or unmute the object.
	 * @return new mute state.
	 */		
	public boolean mute(boolean mute)
	{
		mMuted = mute;
		return mMuted;
	}
	
	/**
	 * Returns statistics on the <code>Channel</code>.
	 * 
	 * @return <code>Stats</code> object containing statistics.
	 */
	public Stats getStatistics()
	{
		return mCurrentStats;
	}
	
	/**
	 * Resets the <code>DestinationObserver</code>.
	 */
	public void reset()
	{
		mCurrentStats.clearStats();
		mStreamOffset = 0;
		mCurrentStats.addStat(BYTE_OFFSET, new Long(mStreamOffset));
	}
	
   /**
    * Used to get the current stream offset.
    * 
    * @return current stream offset
    */
   public synchronized long getStreamOffset()
   {
	   return mStreamOffset;
   }
   
   /**
    * Increments the curent stream offset.
    * 
    * @param dOffset amount (in bytes) to increment the offset by
    */
   private synchronized void incrementStreamOffset(long dOffset)
   {
	   mStreamOffset += dOffset;
   }	
}